<?php
/**
 * Queue Synchronization Script
 * 
 * This script synchronizes message queues between the public directory
 * (where chatapp_webhook.php stores them) and the root directory 
 * (where direct_message_processor.php looks for them)
 */

// Configuration
$public_queue_file = __DIR__ . '/public/message_queue.json';
$root_queue_file = __DIR__ . '/message_queue.json';
$log_file = __DIR__ . '/queue_sync.log';
$sync_interval = 6; // seconds between syncs - increased from 2 to 6 to respect 5 second batching window
$batch_window = 5; // The batching window in seconds (matching webhook's window)

// Make sure the log file exists
if (!file_exists($log_file)) {
    file_put_contents($log_file, "");
}

// Logging function
function sync_log($message) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[$timestamp] $message" . PHP_EOL, FILE_APPEND);
    echo "[$timestamp] $message" . PHP_EOL;
}

sync_log("Queue Sync Service started");
sync_log("Syncing between: $public_queue_file and $root_queue_file");
sync_log("Sync interval: $sync_interval seconds (batching window: $batch_window seconds)");

// Main sync loop
while (true) {
    try {
        // Read from public queue
        if (file_exists($public_queue_file)) {
            $public_queue_content = file_get_contents($public_queue_file);
            $public_queue = json_decode($public_queue_content, true) ?: [];
            
            if (!empty($public_queue)) {
                sync_log("Found " . count($public_queue) . " messages in public queue");
                
                // Check if there are any recent messages that might still be batched
                $now = time();
                $has_recent_messages = false;
                foreach ($public_queue as $msg) {
                    if (($now - ($msg['timestamp'] ?? 0)) < $batch_window) {
                        $has_recent_messages = true;
                        break;
                    }
                }
                
                if ($has_recent_messages) {
                    sync_log("Found messages within batching window - waiting to allow batching");
                    // Skip processing this cycle to allow batching
                    sleep($sync_interval);
                    continue;
                }
                
                // Read current root queue
                $root_queue = [];
                if (file_exists($root_queue_file)) {
                    $root_queue_content = file_get_contents($root_queue_file);
                    $root_queue = json_decode($root_queue_content, true) ?: [];
                }
                
                // Track unique messages by content hash to avoid duplication
                $message_hashes = [];
                foreach ($root_queue as $msg) {
                    // Create a more robust hash using multiple fields
                    $hash = md5(
                        ($msg['dialog_id'] ?? '') . '|' . 
                        ($msg['message'] ?? '') . '|' . 
                        ($msg['customer_phone'] ?? '') . '|' .
                        ($msg['timestamp'] ?? '')
                    );
                    $message_hashes[$hash] = true;
                    
                    // Also track by message ID if available
                    if (isset($msg['id'])) {
                        $message_hashes['id_' . $msg['id']] = true;
                    }
                }
                
                // Add new messages from public queue
                $new_message_count = 0;
                foreach ($public_queue as $key => $message) {
                    // Create a hash from message content to detect duplicates
                    $hash = md5(
                        ($message['dialog_id'] ?? '') . '|' . 
                        ($message['message'] ?? '') . '|' . 
                        ($message['customer_phone'] ?? '') . '|' .
                        ($message['timestamp'] ?? '')
                    );
                    
                    // Also check by message ID if available
                    $id_duplicate = false;
                    if (isset($message['id'])) {
                        $id_duplicate = isset($message_hashes['id_' . $message['id']]);
                    }
                    
                    // Skip duplicates
                    if (isset($message_hashes[$hash]) || $id_duplicate) {
                        sync_log("Skipping duplicate message: " . substr($message['message'] ?? '', 0, 30) . "...");
                        continue;
                    }
                    
                    // Add to root queue
                    $root_queue[] = $message;
                    $message_hashes[$hash] = true;
                    if (isset($message['id'])) {
                        $message_hashes['id_' . $message['id']] = true;
                    }
                    $new_message_count++;
                }
                
                if ($new_message_count > 0) {
                    // Save updated root queue
                    file_put_contents($root_queue_file, json_encode($root_queue, JSON_PRETTY_PRINT));
                    sync_log("Added $new_message_count new messages to root queue");
                }
                
                // Clear the public queue after sync
                file_put_contents($public_queue_file, json_encode([], JSON_PRETTY_PRINT));
                sync_log("Cleared public queue after sync");
            }
        } else {
            sync_log("Public queue file not found, creating empty file");
            file_put_contents($public_queue_file, json_encode([], JSON_PRETTY_PRINT));
        }
    } catch (Exception $e) {
        sync_log("ERROR: " . $e->getMessage());
    }
    
    // Wait before next sync
    sleep($sync_interval);
} 