<?php
// Test script for deal update functionality

// Include necessary files
require_once 'bitrix24/BitrixApiHelper.php';
require_once 'bitrix24/MessageHandlerExtension.php';

// Load configuration
$configPath = __DIR__ . '/config.php';
if (!file_exists($configPath)) {
    die("Configuration file not found at: $configPath\n");
}

$config = include $configPath;

// Initialize the BitrixApiHelper
$apiHelper = new BitrixApiHelper($config['bitrix24']['webhook_url']);

// Initialize PDO for database connection
try {
    $dbConfig = include __DIR__ . '/public/db_config.php';
    
    if ($dbConfig['type'] === 'sqlite') {
        $dsn = "sqlite:{$dbConfig['path']}";
        $pdo = new PDO($dsn);
    } else {
        $dsn = "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']};charset=utf8mb4";
        $pdo = new PDO($dsn, $dbConfig['username'], $dbConfig['password']);
    }
    
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage() . "\n");
}

// Initialize MessageHandlerExtension
$messageHandler = new MessageHandlerExtension($config, $pdo, $apiHelper);

// Test Case 1: Test deal data JSON extraction
function testJsonExtraction($messageHandler) {
    echo "Testing JSON extraction...\n";
    
    $messageText = "Here's your deal information:\n```json\n{\"product\":\"Laptop\",\"quantity\":\"10\",\"company\":\"ABC Tech\",\"city\":\"Riyadh\",\"client_name\":\"John Doe\"}\n```";
    
    $reflectionClass = new ReflectionClass('MessageHandlerExtension');
    $method = $reflectionClass->getMethod('extractJson');
    $method->setAccessible(true);
    
    $result = $method->invoke($messageHandler, $messageText);
    
    if ($result) {
        echo "SUCCESS: Extracted JSON: " . $result . "\n";
    } else {
        echo "FAILED: Could not extract JSON\n";
    }
}

// Test Case 2: Test deal data validation
function testDealDataValidation($messageHandler) {
    echo "Testing deal data validation...\n";
    
    $validData = [
        'product' => 'Laptop',
        'quantity' => '10',
        'company' => 'ABC Tech',
        'city' => 'Riyadh',
        'client_name' => 'John Doe'
    ];
    
    $validMetadataData = [
        'metadata' => [
            'type' => 'bitrix_data',
            'confirmed' => true
        ],
        'customer_data' => [
            'product_name' => 'Laptop',
            'product_quantity' => '10',
            'company' => 'ABC Tech',
            'delivery_city' => 'Riyadh',
            'name' => 'John Doe'
        ]
    ];
    
    $invalidData = [
        'foo' => 'bar',
        'test' => 'data'
    ];
    
    $reflectionClass = new ReflectionClass('MessageHandlerExtension');
    $method = $reflectionClass->getMethod('isValidDealData');
    $method->setAccessible(true);
    
    $result1 = $method->invoke($messageHandler, $validData);
    $result2 = $method->invoke($messageHandler, $validMetadataData);
    $result3 = $method->invoke($messageHandler, $invalidData);
    
    echo "Traditional format validation: " . ($result1 ? "Valid" : "Invalid") . "\n";
    echo "Metadata format validation: " . ($result2 ? "Valid" : "Invalid") . "\n";
    echo "Invalid data validation: " . ($result3 ? "Valid" : "Invalid") . "\n";
}

// Test Case 3: Test deal ID extraction
function testDealIdExtraction($apiHelper, $messageHandler) {
    echo "Testing deal ID extraction...\n";
    
    // Create a mock message
    $mockMessage = [
        'message_id' => '12345',
        'dialog_id' => 'chat12345',
        'user_id' => 'user1',
        'message_text' => 'Test message',
        'entity_data_1' => 'DEAL|54321',
        'customer_phone' => '+9665XXXXXXXX'
    ];
    
    $reflectionClass = new ReflectionClass('MessageHandlerExtension');
    $method = $reflectionClass->getMethod('getDealIdFromMessage');
    $method->setAccessible(true);
    
    $dealId = $method->invoke($messageHandler, $mockMessage);
    
    if ($dealId) {
        echo "SUCCESS: Extracted deal ID: $dealId\n";
    } else {
        echo "FAILED: Could not extract deal ID\n";
    }
}

// Test Case 4: Test full deal processing
function testDealProcessing($messageHandler) {
    echo "Test 4: Full Deal Processing\n";
    
    // Mock a message containing deal data
    $mockMessage = [
        'id' => 9999,
        'message_id' => 'test12345',
        'dialog_id' => 'chat232835',
        'entity_data_1' => 'Y|DEAL|225421|N|N|172433|1744549392|0|0|0',
        'entity_data_2' => 'LEAD|0|COMPANY|0|CONTACT|81993|DEAL|225421',
        'message_text' => <<<EOT
Here is the information for updating the deal:
```json
{
    "type": "bitrix_data",
    "confirmed": true,
    "customer_data": {
        "name": "Mohammed Test",
        "company": "Test Company LLC",
        "product_name": "Test Product",
        "product_quantity": "50",
        "delivery_city": "Riyadh",
        "email": "mohammed@example.com",
        "phone": "+9665XXXXXXXX"
    }
}
```
EOT
    ];
    
    $result = $messageHandler->processDealInformation($mockMessage, [
        'dialog_id' => 'chat232835',
        'deal_id' => 225421
    ]);
    
    if ($result) {
        echo "SUCCESS: Deal information processed\n";
    } else {
        echo "FAILED: Could not process deal information\n";
    }
}

// Run the tests
echo "Starting deal update functionality tests...\n";

testJsonExtraction($messageHandler);
echo "\n";

testDealDataValidation($messageHandler);
echo "\n";

testDealIdExtraction($apiHelper, $messageHandler);
echo "\n";

// Uncomment this to test with a real deal ID
// testDealProcessing($messageHandler);

echo "Tests completed.\n"; 