<?php
// Test script for JSON extraction and deal updates

// Include necessary files
require_once 'bitrix24/BitrixApiHelper.php';
require_once 'bitrix24/MessageHandlerExtension.php';

// Load configuration
$configPath = __DIR__ . '/config.php';
if (!file_exists($configPath)) {
    die("Configuration file not found at: $configPath\n");
}

$config = include $configPath;

// Initialize the BitrixApiHelper
$apiHelper = new BitrixApiHelper($config['bitrix24']['webhook_url']);

// Initialize PDO for database connection
try {
    $dbConfigPath = __DIR__ . '/public/db_config.php';
    if (file_exists($dbConfigPath)) {
        $dbConfig = include $dbConfigPath;
        
        if ($dbConfig['type'] === 'sqlite') {
            $dsn = "sqlite:{$dbConfig['path']}";
            $pdo = new PDO($dsn);
        } else {
            $dsn = "mysql:host={$dbConfig['host']};dbname={$dbConfig['database']};charset=utf8mb4";
            $pdo = new PDO($dsn, $dbConfig['username'], $dbConfig['password']);
        }
        
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } else {
        $pdo = null;
        echo "Warning: Database config not found, some tests may be skipped.\n";
    }
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage() . "\n");
}

// Initialize MessageHandlerExtended
$messageHandler = new MessageHandlerExtended($config, $pdo, $apiHelper);

// Test 1: Test entity_data extraction
echo "Test 1: Entity Data Extraction\n";
$entityData1 = "Y|DEAL|225421|N|N|172433|1744549392|0|0|0";
$entityData2 = "LEAD|0|COMPANY|0|CONTACT|81993|DEAL|225421";

$dealId1 = $apiHelper->extractDealId($entityData1);
$dealId2 = $apiHelper->extractDealId($entityData2);

echo "Entity Data 1: $entityData1 -> Deal ID: " . ($dealId1 ?: "Not found") . "\n";
echo "Entity Data 2: $entityData2 -> Deal ID: " . ($dealId2 ?: "Not found") . "\n";

// Test 2: Test JSON extraction from various formats
echo "\nTest 2: JSON Extraction\n";

$testMessages = [
    // Test 1: Regular JSON in code block
    "Here's the deal information:\n```json\n{\"metadata\":{\"type\":\"bitrix_data\",\"confirmed\":true},\"customer_data\":{\"name\":\"عزام\",\"company\":\"سيكو\",\"email\":\"Azzam@gmail.com\",\"phone\":\"056789988\",\"product_name\":\"براميل\",\"product_quantity\":\"150000\",\"delivery_city\":\"جدة\"}}\n```",
    
    // Test 2: Escaped JSON 
    "Here's the deal information: {\\\"metadata\\\":{\\\"type\\\":\\\"bitrix_data\\\",\\\"confirmed\\\":true},\\\"customer_data\\\":{\\\"name\\\":\\\"عزام\\\",\\\"company\\\":\\\"سيكو\\\",\\\"email\\\":\\\"Azzam@gmail.com\\\",\\\"phone\\\":\\\"056789988\\\",\\\"product_name\\\":\\\"براميل\\\",\\\"product_quantity\\\":\\\"150000\\\",\\\"delivery_city\\\":\\\"جدة\\\"}}",
    
    // Test 3: Unicode escaped JSON 
    "Here's the deal information: {\u0022metadata\u0022:{\u0022type\u0022:\u0022bitrix_data\u0022,\u0022confirmed\u0022:true},\u0022customer_data\u0022:{\u0022name\u0022:\u0022عزام\u0022,\u0022company\u0022:\u0022سيكو\u0022,\u0022email\u0022:\u0022Azzam@gmail.com\u0022,\u0022phone\u0022:\u0022056789988\u0022,\u0022product_name\u0022:\u0022براميل\u0022,\u0022product_quantity\u0022:\u0022150000\u0022,\u0022delivery_city\u0022:\u0022جدة\u0022}}",
    
    // Test 4: JSON with linebreaks
    "Here's the deal information: 
    {
        \"metadata\": {
            \"type\": \"bitrix_data\",
            \"confirmed\": true
        },
        \"customer_data\": {
            \"name\": \"عزام\",
            \"company\": \"سيكو\",
            \"email\": \"Azzam@gmail.com\",
            \"phone\": \"056789988\",
            \"product_name\": \"براميل\",
            \"product_quantity\": \"150000\",
            \"delivery_city\": \"جدة\"
        }
    }"
];

// Create a simple JSON extraction function instead of using reflection
function extractJsonForTest($text) {
    echo "[" . date('D M d H:i:s Y') . "] TEST: Attempting to extract JSON" . PHP_EOL;
    
    // Handle escaped JSON first
    $text = str_replace(['\\u0022', '\\"'], '"', $text);
    $text = stripslashes($text);
    
    // Method 1: Try to extract JSON from code blocks with ```json syntax
    if (preg_match('/```(?:json)?\s*(.*?)\s*```/s', $text, $matches)) {
        echo "[" . date('D M d H:i:s Y') . "] TEST: Found JSON in code block format" . PHP_EOL;
        return $matches[1];
    }
    
    // Method 2: Try to extract JSON with recursive brace matching - most comprehensive pattern
    if (preg_match('/{(?:[^{}]|(?R))*}/s', $text, $matches)) {
        echo "[" . date('D M d H:i:s Y') . "] TEST: Found JSON with brace matching" . PHP_EOL;
        return $matches[0];
    }
    
    // Method 3: Try to find any JSON-like content with a more relaxed pattern
    if (preg_match('/{.+?}/s', $text, $matches)) {
        echo "[" . date('D M d H:i:s Y') . "] TEST: Found JSON with relaxed pattern" . PHP_EOL;
        return $matches[0];
    }
    
    echo "[" . date('D M d H:i:s Y') . "] TEST: No JSON found in message" . PHP_EOL;
    return null;
}

foreach ($testMessages as $i => $message) {
    echo "Testing message format $i:" . PHP_EOL;
    $result = extractJsonForTest($message);
    echo "Result: " . (strlen($result) > 0 ? "Extracted " . strlen($result) . " chars" : "Failed") . "\n";
    if ($result) {
        $json = json_decode($result, true);
        echo "  Valid JSON: " . (json_last_error() === JSON_ERROR_NONE ? "Yes" : "No - " . json_last_error_msg()) . "\n";
    }
}

// Test 3: Deal Data Validation
echo "\nTest 3: Deal Data Validation\n";

$testData = [
    [
        'metadata' => ['type' => 'bitrix_data', 'confirmed' => true],
        'customer_data' => [
            'name' => 'Test Name',
            'company' => 'Test Company',
            'product_name' => 'Test Product',
            'product_quantity' => '100',
            'delivery_city' => 'Test City'
        ]
    ],
    [
        'product' => 'Direct Product',
        'quantity' => '200',
        'company' => 'Direct Company',
        'city' => 'Direct City'
    ],
    [
        'deal_id' => '123456',
        'some_other_field' => 'test'
    ],
    [
        'random_data' => 'This should not be valid'
    ]
];

// Create a direct validation function
function isValidDealDataTest($data) {
    echo "[" . date('D M d H:i:s Y') . "] TEST: Validating deal data" . PHP_EOL;
    
    // First check for bitrix_data format with customer_data
    if (isset($data['metadata']) && isset($data['metadata']['type'])) {
        $dataType = strtolower($data['metadata']['type']);
        if ($dataType === 'bitrix_data' && isset($data['customer_data'])) {
            $confirmed = $data['metadata']['confirmed'] ?? true; // Default to true if not specified
            
            if ($confirmed) {
                echo "[" . date('D M d H:i:s Y') . "] TEST: Detected valid bitrix_data format with customer_data" . PHP_EOL;
                return true;
            }
        }
        
        // Also accept other metadata types
        if (in_array($dataType, ['deal_data', 'deal_info', 'deal_update', 'deal'])) {
            echo "[" . date('D M d H:i:s Y') . "] TEST: Detected valid deal metadata type: $dataType" . PHP_EOL;
            return true;
        }
    }
    
    // Check if customer_data exists directly at the root
    if (isset($data['customer_data']) && is_array($data['customer_data'])) {
        echo "[" . date('D M d H:i:s Y') . "] TEST: Found customer_data at root level" . PHP_EOL;
        return true;
    }
    
    // Then check for essential deal fields in the traditional format
    $requiredFields = ['product', 'company', 'city', 'quantity', 'client_name', 'product_name', 'delivery_city', 'name'];
    $foundFields = [];
    
    foreach ($requiredFields as $field) {
        if (isset($data[$field]) && !empty($data[$field])) {
            $foundFields[] = $field;
        }
    }
    
    // If we found any required fields
    if (!empty($foundFields)) {
        echo "[" . date('D M d H:i:s Y') . "] TEST: Found required deal fields: " . implode(', ', $foundFields) . PHP_EOL;
        return true;
    }
    
    // Last check: general fields that indicate deal data
    $dealIndicatorFields = ['deal_id', 'deal', 'product', 'quantity', 'company', 'client'];
    foreach ($dealIndicatorFields as $field) {
        if (isset($data[$field])) {
            echo "[" . date('D M d H:i:s Y') . "] TEST: Found deal indicator field: $field" . PHP_EOL;
            return true;
        }
    }
    
    echo "[" . date('D M d H:i:s Y') . "] TEST: Deal data validation result: Invalid - no recognizable deal data" . PHP_EOL;
    return false;
}

foreach ($testData as $i => $data) {
    echo "Testing data format $i: " . PHP_EOL;
    $result = isValidDealDataTest($data);
    echo "Result: " . ($result ? "Valid deal data" : "Not valid deal data") . "\n";
}

// Test 4: Full Message Processing
echo "\nTest 4: Full Message Processing\n";

/*
// This test is disabled because it requires a fully initialized MessageHandler
$mockMessage = [
    'id' => 9999,
    'message_id' => 'test12345',
    'dialog_id' => 'chat232835',
    'entity_data_1' => 'Y|DEAL|225421|N|N|172433|1744549392|0|0|0',
    'entity_data_2' => 'LEAD|0|COMPANY|0|CONTACT|81993|DEAL|225421',
    'message_text' => "{\"metadata\":{\"type\":\"bitrix_data\",\"confirmed\":true},\"customer_data\":{\"name\":\"Test Name\",\"company\":\"Test Company\",\"email\":\"test@example.com\",\"phone\":\"1234567890\",\"product_name\":\"Test Product\",\"product_quantity\":\"100\",\"delivery_city\":\"Test City\"}}"
];

// Use the public method directly
$result = $messageHandler->processDealInformation($mockMessage, [
    'dialog_id' => 'chat232835',
    'deal_id' => 225421
]);
echo "Deal update result: " . ($result ? "SUCCESS" : "FAILED") . "\n";
*/

echo "Skipping full message processing test - run manually through the web interface\n";

// Test 5: Testing direct deal update via API
echo "\nTest 5: Direct Deal Update\n";

$dealData = [
    'product' => 'Test Product from Debug Script',
    'company' => 'Test Company',
    'city' => 'Test City',
    'quantity' => '999',
    'client_name' => 'Test Client'
];

echo "[" . date('D M d H:i:s Y') . "] TEST: Attempting direct deal update via API" . PHP_EOL;
$updateResult = $apiHelper->updateDeal(225421, $dealData);
$success = isset($updateResult['result']) && $updateResult['result'];

echo "Direct deal update result: " . ($success ? "SUCCESS" : "FAILED") . "\n";
if (!$success) {
    echo "Error: " . json_encode($updateResult) . "\n";
}

echo "\nAll tests completed.\n"; 